#ifndef __CNativeImage__
#define __CNativeImage__

#include <MathTools/CDimension.hpp>
#include <MathTools/CRect.hpp>
#include "../Windowing/SWindowHandle.hpp"
#include "../Basics/CColour.hpp"
#include "SGraphicsHandle.hpp"

//	===========================================================================

using Exponent::MathTools::CDimension;
using Exponent::MathTools::CRect;
using Exponent::GUI::Windowing::SWindowHandle;
using Exponent::GUI::Basics::CColour;
using Exponent::GUI::Graphics::SGraphicsHandle;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Graphics
		{
			/**
			 * @class CNativeImage CNativeImage.hpp
			 * @brief An offscreen draw context
			 *
			 * Wraps up a native image in a class. Used by CGraphics \n
			 * to implement an effective double buffering routine
			 *
			 * @date 01/10/2004
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CNativeImage.hpp,v 1.3 2007/02/08 21:07:54 paul Exp $
			 */
			class CNativeImage
			{
			public:

				/**
				 * Construction
				 */
				CNativeImage();

				/**
				 * Destruction
				 */
				virtual ~CNativeImage();

//	===========================================================================

				/**
				 * Get the bitmap graphics handle
				 * @retval SGraphicsHandle* The graphics handle
				 */
				SGraphicsHandle *getGraphicsHandle() { return &m_graphicsHandle; }

				/**
				 * Get the window handle associated with teh draw context
				 * @retval SWindowHandle* The window handle
				 */
				SWindowHandle *getWindowHandle() const { return m_windowHandle; }

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					/**
					 * Get the parent DC context
					 * @retval HDC The windows draw context
					 */
					HDC getParentContext() const { return m_parentHDC; }
				/** @cond */
				#endif
				/** @endcond */

//	===========================================================================

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					/**
					 * Initialise the image
					 * @param windowHandle Handle to the parent window
					 * @param drawContext The windows draw context
					 * @param area The size of the image to create
					 */
					void initialise(SWindowHandle *windowHandle, HDC drawContext, const CDimension &area);
				/** @cond */
				#else
				/** @endcond */
					/**
					 * Initialise the image
					 * @param windowHandle Handle to the parent window
					 * @param area The size of the image to create
					 */
					void initialise(SWindowHandle *windowHandle, const CDimension &area);
				/** @cond */
				#endif
				/** @endcond */

				/**
				 * Deinitialise
				 */
				void uninitialise();

//	===========================================================================

				/**
				 * Copy this to a neew native window
				 * @param image On return this is a copy of this image
				 */
				void copyTo(CNativeImage *image);

				/**
				 * Copy from another image to this
				 * @param image The image to copy
				 */
				void copyFrom(CNativeImage *image);

				/**
				 * Alpha blend on to another image
				 * @param image The image to blend
				 * @param where Where on the source image are we copying from
				 * @param toHere The position on this image to beldn to
				 * @param alpha The alpha blend level as a normalised double
				 */
				void alphaBlendTo(CNativeImage *image, const CRect &where, const CRect &toHere, const double alpha);

//	===========================================================================

				/**
				 * Set the background colour
				 * @param colour The colour to use at the next initialisation for the background colour
				 */
				void setBackgroundColour(const CColour &colour) { m_backgroundColour = colour; }

				/**
				 * Get the background colour
				 * @retval const CColour& The image background colour
				 */
				const CColour &getBackgroundColour() const { return m_backgroundColour; }

				/**
				 * Get the dimension of the image
				 * @retval const CDimension& The dimension of the image
				 */
				const CDimension &getDimensions() const { return m_area; }

//	===========================================================================

			protected:

//	===========================================================================

				/**
				 * Create the native image
				 */
				void createNativeImage();

				/**
				 * Destroy the native image
				 */
				void destroyNativeImage();

//	===========================================================================

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					HDC m_parentHDC;				/**< The parent HDC */
				/** @cond */
				#endif
				/** @endcond */

				SGraphicsHandle m_graphicsHandle;	/**< Handle to the graphics contexts */
				SWindowHandle *m_windowHandle;		/**< Handle to the window your drawing into */
				CDimension m_area;					/**< The area to draw in */
				CColour m_backgroundColour;			/**< The background colour */
				bool m_initialised;					/**< Is this initialised */
			};
		}
	}
}
#endif	// End of CNativeImage.hpp